// Author: A.Voss@FH-Aachen.de
//
// class operators 
// tags: operators

// http://stackoverflow.com/questions/4421706/operator-overloading/4421729#4421729
//
// The Three Basic Rules of Operator Overloading in C++
// - Whenever the meaning of an operator is not obviously clear and undisputed, 
//   it should not be overloaded. 
// - Always stick to the operator’s well-known semantics.
// - Always provide all out of a set of related operations.
//
// The Decision between Member and Non-member
// - The binary operators = (assignment), [] (array subscription), -> (member access), 
//   as well as the n- ary () (function call) operator, must always be implemented as member functions [...]
// For all operators where you have to choose to either implement them as a member function 
// or a non- member function, use the following rules of thumb to decide:
// - If it is a unary operator, implement it as a member function.
// - If a binary operator treats both operands equally (it leaves them unchanged), 
//   implement this operator as a non-member function.
// - If a binary operator does not treat both of its operands equally (usually it will 
//   change its left operand), it might be useful to make it a member function of its 
//   left operand’s type, if it has to access the

#pragma GCC diagnostic ignored "-Wunused-variable"
#pragma GCC diagnostic ignored "-Wunused-but-set-variable"

#include <iostream>
#include <cstdlib>
#include <string>
#include <cmath>
using namespace std;

#include "../../include/tools.h"

class Line  // a*x+b
{
    double a,b;
    
public: 
    Line(double A, double B) : a(A), b(B) { }
    
    Line(const Line & l) : Line(l.a,l.b) { }    // C++11, use one ctor

    Line & operator=(Line rhs)  // use swap
    {
        // swap(rhs);
        std::swap(a,rhs.a);
        std::swap(b,rhs.b);
        return *this;
    }
    
    double operator()(double x) // eval function
    { 
        return a*x+b;
    }

    friend bool operator==(const Line& lhs, const Line& rhs)
    {
        return fabs(lhs.a-rhs.a)<1e-10 && fabs(lhs.b-rhs.b)<1e-10;
    }
    
    friend bool operator!=(const Line& lhs, const Line& rhs)
    {
        return !operator==(lhs,rhs);    // use op==
    }
     
    Line & operator++()     // prefix
    {
        a+=1.0; b+=1.0;
        return *this;
    }
    
    Line operator++(int)    // postfix
    {
        Line tmp(*this);
        operator++();       // use prefix
        return tmp;
    }
    
    Line & operator+=(const Line & rhs)
    {
        a += rhs.a;     // arbitrary operation ...
        b += rhs.b;
        return *this;
    }
    
    friend Line operator+(Line lhs, const Line & rhs)   // note copy-ctor
    {
      lhs += rhs;       // use op+=
      return lhs;
    }
    
    double & operator[](int n)
    {
        return (0==n) ? b : a;
    }

    const double & operator[](int n) const
    {
        return (0==n) ? b : a;
    }

    friend ostream & operator<<(ostream & os, const Line & p)
    {
        os << "(" << p.a << "*x+" << p.b << ")";
        return os;
    }
};

int main()
{
    tools_log(); cout << "play with operators ..." << endl;

    Line l1(1,2);
    Line l2(2,3);
    
    double y;
    bool b;
    
    l2 = l1;
    
    y = l2(3);
    
    b = (l1==l2);
    b = (l1!=l2);
    
    l1++;
    ++l1;
    
    l1 += l2;
    l1 = l1+l2;
    
    return EXIT_SUCCESS;
}

